"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.lookupTableProvider = void 0;
const web3_js_1 = require("@solana/web3.js");
const config_1 = require("../../config");
class LookupTableProvider {
    constructor() {
        this.lookupTables = new Map();
        this.lookupTablesForAddress = new Map();
        this.addressesForLookupTable = new Map();
    }
    updateCache(lutAddress, lutAccount) {
        this.lookupTables.set(lutAddress.toBase58(), lutAccount);
        this.addressesForLookupTable.set(lutAddress.toBase58(), new Set());
        for (const address of lutAccount.state.addresses) {
            const addressStr = address.toBase58();
            this.addressesForLookupTable.get(lutAddress.toBase58()).add(addressStr);
            if (!this.lookupTablesForAddress.has(addressStr)) {
                this.lookupTablesForAddress.set(addressStr, new Set());
            }
            this.lookupTablesForAddress.get(addressStr).add(lutAddress.toBase58());
        }
    }
    processLookupTableUpdate(lutAddress, data) {
        const lutAccount = new web3_js_1.AddressLookupTableAccount({
            key: lutAddress,
            state: web3_js_1.AddressLookupTableAccount.deserialize(data.data),
        });
        this.updateCache(lutAddress, lutAccount);
        return;
    }
    async getLookupTable(lutAddress) {
        const lutAddressStr = lutAddress.toBase58();
        if (this.lookupTables.has(lutAddressStr)) {
            return this.lookupTables.get(lutAddressStr);
        }
        const lut = await config_1.connection.getAddressLookupTable(lutAddress);
        if (lut.value === null) {
            return null;
        }
        this.updateCache(lutAddress, lut.value);
        return lut.value;
    }
    async createLookupTable(authority, payer) {
        const recentSlot = await config_1.connection.getSlot();
        const [lookupTableIx, lookupTableAddress] = web3_js_1.AddressLookupTableProgram.createLookupTable({
            authority,
            recentSlot,
            payer,
        });
        const transaction = new web3_js_1.Transaction().add(lookupTableIx);
        const signature = await config_1.connection.sendTransaction(transaction, [web3_js_1.Keypair.generate()]);
        await config_1.connection.confirmTransaction(signature);
        console.log('Created Address Lookup Table:', lookupTableAddress.toBase58());
        return lookupTableAddress;
    }
    computeIdealLookupTablesForAddresses(addresses) {
        const MIN_ADDRESSES_TO_INCLUDE_TABLE = 2;
        const MAX_TABLE_COUNT = 3;
        const addressSet = new Set();
        const tableIntersections = new Map();
        const selectedTables = [];
        const remainingAddresses = new Set();
        let numAddressesTakenCareOf = 0;
        for (const address of addresses) {
            const addressStr = address.toBase58();
            if (addressSet.has(addressStr))
                continue;
            addressSet.add(addressStr);
            const tablesForAddress = this.lookupTablesForAddress.get(addressStr) || new Set();
            if (tablesForAddress.size === 0)
                continue;
            remainingAddresses.add(addressStr);
            for (const table of tablesForAddress) {
                const intersectionCount = tableIntersections.get(table) || 0;
                tableIntersections.set(table, intersectionCount + 1);
            }
        }
        const sortedIntersectionArray = Array.from(tableIntersections.entries()).sort((a, b) => b[1] - a[1]);
        for (const [lutKey, intersectionSize] of sortedIntersectionArray) {
            if (intersectionSize < MIN_ADDRESSES_TO_INCLUDE_TABLE)
                break;
            if (selectedTables.length >= MAX_TABLE_COUNT)
                break;
            if (remainingAddresses.size <= 1)
                break;
            const lutAddresses = this.addressesForLookupTable.get(lutKey);
            const addressMatches = new Set([...remainingAddresses].filter((x) => lutAddresses.has(x)));
            if (addressMatches.size >= MIN_ADDRESSES_TO_INCLUDE_TABLE) {
                selectedTables.push(this.lookupTables.get(lutKey));
                for (const address of addressMatches) {
                    remainingAddresses.delete(address);
                    numAddressesTakenCareOf++;
                }
            }
        }
        return selectedTables;
    }
}
const lookupTableProvider = new LookupTableProvider();
exports.lookupTableProvider = lookupTableProvider;
